####################################################################
# Curses stuff

import curses, string

class CodeWindow:
	# Member variables
	# x1, y1, width, height	-- Window coordinates
	# cwnd			-- Curses window
	# disasm		-- A Disassembler object
	# wndlines		-- An array of the lines of display in the window
	# wndpa			-- Top-of-window program address
	# pc			-- Current program counter
	# spa			-- Selected program address
	def __init__(self, cwnd, x, y, w, h):
		self.cwnd = cwnd
		self.x, self.y, self.width, self.height = x,y,w,h
		self.disasm = None
		cwnd.border()
		cwnd.addstr(0, 2, "[Code]")
		
		self.wndpa = 0
		self.pc = 0
		self.spa = 0
		self.wndlines = [ ]

	def set_display_params(self, wndpa, pc, spa):
		self.wndpa = wndpa
		self.pc = pc
		self.spa = spa

	def update_pc(self, pc):
		self.pc = pc
		if self.pc < self.wndpa or self.pc >= (self.wndpa+self.height-2):
			self.wndpa = self.pc - 8
		self.spa = self.pc
		
	def move_cursor(self, pa):
		self.spa = pa
		self.repaint()
		
	def move_cursor_up(self):
		self.spa = self.spa - 2
		self.repaint()

	def move_cursor_down(self):
		self.spa = self.spa + 2
		self.repaint()
		
	def disasm_one_line(self, addr):
		da = self.disasm.disasm(addr)
		if da[:5] == '.word':
			dap = string.split(da)
			da = dap[0] + '\t' + dap[1]
		chars = list(self.disasm.read(addr, 2))
		cv1 = ord(chars[0])
		cv2 = ord(chars[1])
		if ord(chars[0]) < 32 or ord(chars[0]) > 126:
			chars[0] = '.'
		if ord(chars[1]) < 32 or ord(chars[1]) > 126:
			chars[1] = '.'
		s = "%08x  %c%c  %02x%02x  %s" % (addr, chars[0], chars[1], \
			cv2, cv1,
			da)
		s = string.replace(s, "\t", " "*4)
		if len(s) >= (self.width-2):
			s = s[:self.width-2]
		elif len(s) < (self.width-2):
			s = s + " "*((self.width - 2) - len(s))
		return s

	def refresh(self):
		self.wndlines = [ ]
		for i in range(0, self.height - 2):
			self.wndlines.append(self.disasm_one_line(self.wndpa+i*2))
		
		self.repaint()

	def repaint(self):
		cw = self.cwnd
		for i in range(0, self.height - 2):
			s = self.wndlines[i]
			if self.wndpa+i*2 == self.pc:
				s = string.replace(s, " ", "*")
			if self.wndpa+i*2 == self.spa:
				cw.standout()
				cw.addstr(i+1, 1, s)
				cw.standend()
			else:
				cw.addstr(i+1, 1, s)
		cw.refresh()

		
class RegistersWindow:
	# Member variables
	# x1, y1, width, height	-- Window coordinates
	# cwnd			-- Curses window
	def __init__(self, cwnd, x, y, w, h):
		self.cwnd = cwnd
		self.x, self.y, self.width, self.height = x,y,w,h
		cwnd.border()
		cwnd.addstr(0, 2, "[Regs]")

	def doodle(self, regs):
		for i in range(16):
			if i<10:
				s = " "
			else:
				s = ""
			self.cwnd.addstr(1+i, 1, "R%d  %s%08x" % (i,s,regs['R%d' % i]))
		self.cwnd.addstr(18, 1, "PC   %08x" % regs['PC'])
		self.cwnd.addstr(19, 1, "PR   %08x" % regs['PR'])
		self.cwnd.addstr(20, 1, "SR   %08x" % regs['SR'])
		self.cwnd.addstr(21, 1, "  t      %d" % (regs['SR'] & 1))
		self.cwnd.addstr(22, 1, "  imask  0000" )
		self.cwnd.addstr(23, 1, "  md     0")
		self.cwnd.addstr(24, 1, "  rb     0")
		self.cwnd.addstr(25, 1, "  bl     0")
		self.cwnd.addstr(26, 1, "  fd     0")
		self.cwnd.addstr(27, 1, "GBR  %08x" % regs['GBR'])
		self.cwnd.addstr(28, 1, "VBR  %08x" % regs['VBR'])
		self.cwnd.addstr(29, 1, "MACH %08x" % regs['MACH'])
		self.cwnd.addstr(30, 1, "MACL %08x" % regs['MACL'])
		self.cwnd.refresh()

class MessageWindow:
	# Member variables
	# x1, y1, width, height	-- Window coordinates
	# cwnd			-- Curses window
	# lines			-- Lines of display text
	# linecnt, colcnt	-- Number of lines, cols of text
	
	def __init__(self, cwnd, x, y, w, h):
		self.cwnd = cwnd
		self.x, self.y, self.width, self.height = x,y,w,h
		cwnd.border()
		cwnd.addstr(0, 2, "[Messages]")
		
		self.linecnt = self.height - 2
		self.colcnt = self.width - 2
		self.lines = [ ]
		for i in range(self.linecnt):
			self.lines.append("")

	def write_message(self, s):
		if len(s) <= self.colcnt:
			self.write_one_line(s)
		else:
			while len(s) > 0:
				sp = s[:self.colcnt]
				self.write_one_line(sp)
				s = s[self.colcnt:]
		self.refresh()
		
	def write_one_line(self, line):
		self.lines = self.lines[1:] + [line]
	
	def refresh(self):
		for i in range(self.linecnt):
			#self.cwnd.addstr(i+1, 1, " " * self.colcnt)
			self.cwnd.hline(i+1, 1, 32, self.colcnt)
			self.cwnd.addstr(i+1, 1, self.lines[i])
		self.cwnd.refresh()


class Screen:
	# Member variables
	# scrn			-- Curses window object of the whole screen
	# scrn_w, scrn_h	-- Integer width and height of screen
	# codewnd		-- Code display window
	# regswnd		-- Register display window
	# msgswnd		-- Messages display window

	def __init__(self):
		self.scrn = curses.initscr()
		curses.noecho()
		self.scrn.keypad(1)
		self.scrn_h, self.scrn_w = self.scrn.getmaxyx()
		self.wnd = { }
		self.draw_menubar()
		self.setup_code_wnd()
		self.setup_regs_wnd()
		self.setup_msgs_wnd()
		self.scrn.refresh()

	def shutdown(self):
		curses.endwin()

	def draw_menubar(self):
		mbtext = " F1-Jump To Address   F2-Jump To Cursor   F5-Go   F6-Auto-step   F7-Step   F10-Modify"
		mbtext = mbtext + " "*(self.scrn_w - len(mbtext))
		self.scrn.standout()
		self.scrn.addstr(0, 0, mbtext)
		self.scrn.standend()

	def setup_code_wnd(self):
		w = self.scrn_w - 20
		h = self.scrn_h - 8
		x = 0
		y = 1
		cwnd = self.scrn.subwin(h, w, y, x)
		self.codewnd = CodeWindow(cwnd, x, y, w, h)
			
	def setup_regs_wnd(self):
		w = 21
		h = self.scrn_h - 8
		x = self.scrn_w - 21
		y = 1
		cwnd = self.scrn.subwin(h, w, y, x)
		self.regswnd = RegistersWindow(cwnd, x, y, w, h)
	
	def setup_msgs_wnd(self):
		w = self.scrn_w
		h = 8
		x = 0
		y = self.scrn_h - 8
		cwnd = self.scrn.subwin(h, w, y, x)
		self.msgswnd = MessageWindow(cwnd, x, y, w, h)
		

import Disassembler
def test_screen():
	s = Screen()
	d = Disassembler.Disassembler()
	s.run(d)

