/* KallistiOS 0.6

   fs.c
   (c)2000 Dan Potter

*/

static char id[] = "KOS $Id: fs.c,v 1.4 2000/11/11 23:32:14 bard Exp $";

/*

This module manages all of the file system code. Basically the VFS works
something like this:

- The kernel contains path primitives. There is a table of VFS path handlers
  installed by loaded servers. When the kernel needs to open a file, it will
  search this path handler table from the bottom until it finds a handler
  that is willing to take the request. The request is then handed off to
  the handler.
- The path handler receives the part of the path that is left after the
  part in the handler table. The path handler should return an internal
  handle for accessing the file. An internal handle of zero is always 
  assumed to mean failure.
- The kernel open function takes this value and wraps it in a structure that
  describes which service handled the request, and its internal handle.
- Subsequent operations go through this abstraction layer to land in the
  right place.

*/

#include <stdio.h>
#include <malloc.h>
#include <kallisti/fs.h>
#include <kallisti/thread.h>

/* Thread mutex */
static thd_mutex_t mutex;

/* File handler structure; this struct contains path/function pairs that 
   describe how to handle a given path name. This is used only during
   the open process. */
typedef struct fs_handler_str {
	char		prefix[MAX_FN_LEN];
	vfs_handler	*handler;
	struct fs_handler_str *next;
} fs_handler_t;
static fs_handler_t *fs_handlers;

/* File handle structure; this is an entirely internal structure so it does
   not go in a header file. */
typedef struct {
	vfs_handler	*handler;	/* Handler */
	uint32		hnd;		/* Handler-internal */
} fs_hnd_t;


/* Internal file commands for root dir reading */
static uint32 fs_root_opendir() {
	fs_hnd_t	*hnd = malloc(sizeof(fs_hnd_t));
	hnd->handler = NULL;
	hnd->hnd = 0;
	return (uint32)hnd;
}

/* Not thread-safe right now */
static dirent_t root_readdir_dirent;
static dirent_t *fs_root_readdir(uint32 hnd) {
	fs_hnd_t	*handle = (fs_hnd_t*)hnd;
	fs_handler_t	*handler = fs_handlers;
	int 		cnt = handle->hnd;
	
	while (handler != NULL && cnt--)
		handler = handler->next;
	
	if (handler == NULL)
		return NULL;

	root_readdir_dirent.size = -1;
	strcpy(root_readdir_dirent.name, handler->prefix);
	handle->hnd++;
	return &root_readdir_dirent;
}

/* Locate a file handler for a given path name */
static fs_handler_t *fs_get_handler(const char *fn) {
	fs_handler_t	*cur = fs_handlers;

	/* Scan the handler table and look for a path match */
	while (cur != NULL) {
		if (!strnicmp(cur->prefix, fn, strlen(cur->prefix)))
			break;
		cur = cur->next;
	}
	if (cur == NULL) {
		/* Couldn't find a handler */
		return NULL;
	}
	else
		return cur;
}

/* Attempt to open a file, given a path name. Follows the process described
   in the above comments. */
uint32 fs_open(char *fn, int mode) {
	fs_handler_t	*cur;
	char		*cname;
	uint32		h;
	fs_hnd_t	*hnd;

	/* Are they trying to open the root? */
	if (!strcmp(fn, "/")) {
		if ((mode & O_DIR)) 
			return fs_root_opendir();
		else
			return 0;
	}

	/* Look for a handler */
	cur = fs_get_handler(fn);
	if (cur == NULL)
		return 0;

	/* Found one -- get the "canonical" path name */
	cname = fn + strlen(cur->prefix);
	/* printf("Got a handler, canonical name is '%s'\r\n", cname); */

	/* Invoke the handler */
	if (cur->handler->open == NULL) return 0;
	/*printf("Handler for open is ok\r\n");*/
	h = cur->handler->open(cname, mode);
	if (h == 0) return 0;
	/*printf("Past check\r\n");*/

	/* Wrap it up in a structure */
	hnd = malloc(sizeof(fs_hnd_t));
	if (hnd == NULL) {
		cur->handler->close(h);
		return 0;
	}
	hnd->handler = cur->handler;
	hnd->hnd = h;

	return (uint32)hnd;
}

/* Close a file and clean up the handle */
void fs_close(uint32 hnd) {
	fs_hnd_t *h = (fs_hnd_t*)hnd;

	if (hnd == 0) return;
	if (h->handler != NULL) {
		if (h->handler->close == NULL) return;
		h->handler->close(h->hnd);
	}
	free(h);
}

/* The rest of these pretty much map straight through */
ssize_t fs_read(uint32 hnd, void *buffer, size_t cnt) {
	fs_hnd_t *h = (fs_hnd_t*)hnd;

	if (hnd == 0 || h->handler == NULL || h->handler->read == NULL) return -1;
	return h->handler->read(h->hnd, buffer, cnt);
}

ssize_t fs_write(uint32 hnd, void *buffer, size_t cnt) {
	fs_hnd_t *h = (fs_hnd_t*)hnd;

	if (hnd == 0 || h->handler == NULL || h->handler->write == NULL) return -1;
	return h->handler->write(h->hnd, buffer, cnt);
}

off_t fs_seek(uint32 hnd, off_t offset, int whence) {
	fs_hnd_t *h = (fs_hnd_t*)hnd;

	if (hnd == 0 || h->handler == NULL || h->handler->seek == NULL) return -1;
	return h->handler->seek(h->hnd, offset, whence);
}

off_t fs_tell(uint32 hnd) {
	fs_hnd_t *h = (fs_hnd_t*)hnd;

	if (hnd == 0 || h->handler == NULL || h->handler->tell == NULL) return -1;
	return h->handler->tell(h->hnd);
}

size_t fs_total(uint32 hnd) {
	fs_hnd_t *h = (fs_hnd_t*)hnd;
	size_t rv;

	if (hnd == 0 || h->handler == NULL || h->handler->total == NULL) return -1;
	return h->handler->total(h->hnd);
}

dirent_t *fs_readdir(uint32 hnd) {
	fs_hnd_t *h = (fs_hnd_t*)hnd;

	if (hnd == 0) return NULL;
	if (h->handler == NULL) return fs_root_readdir(hnd);
	if (h->handler->readdir == NULL) return NULL;
	return h->handler->readdir(h->hnd);
}

int fs_ioctl(uint32 hnd, void *data, size_t size) {
	fs_hnd_t *h = (fs_hnd_t*)hnd;

	if (hnd == 0 || h->handler == NULL || h->handler->readdir == NULL) return -1;
	return h->handler->ioctl(h->hnd, data, size);
}

int fs_rename(const char *fn1, const char *fn2) {
	fs_handler_t	*fh1, *fh2;
	
	/* Look for handlers */
	fh1 = fs_get_handler(fn1);
	if (fh1 == NULL) return -1;
	
	fh2 = fs_get_handler(fn2);
	if (fh2 == NULL) return -1;
	
	if (fh1 != fh2) return -2;

	if (fh1->handler->rename)
		return fh1->handler->rename(fn1+strlen(fh1->prefix),
			fn2+strlen(fh1->prefix));
	else
		return -3;
}

int fs_unlink(const char *fn) {
	fs_handler_t	*cur;
	
	/* Look for a handler */
	cur = fs_get_handler(fn);
	if (cur == NULL) return 1;

	if (cur->handler->unlink)	
		return cur->handler->unlink(fn+strlen(cur->prefix));
	else
		return -1;
}



/* Add a VFS module */
int fs_handler_add(char *prefix, vfs_handler *hnd) {
	fs_handler_t *fshnd;

	fshnd = malloc(sizeof(fs_handler_t));
	if (fshnd == NULL)
		return -1;

	thd_mutex_lock(&mutex);

	strcpy(fshnd->prefix, prefix);
	fshnd->handler = hnd;
	fshnd->next = fs_handlers;
	fs_handlers = fshnd;

	thd_mutex_unlock(&mutex);

	return 0;
}

/* Remove a VFS module */
int fs_handler_remove(vfs_handler *hnd) {
	fs_handler_t *last, *cur;

	thd_mutex_lock(&mutex);

	last = NULL;
	cur = fs_handlers;
	while (cur != NULL) {
		if (cur->handler == hnd) {
			break;
		}
		last = cur;
		cur = cur->next;
	}
	if (cur == NULL) {
		thd_mutex_unlock(&mutex);
		return -1;
	}

	if (last == NULL)
		fs_handlers = cur->next;
	else
		last->next = cur->next;
	free(cur);

	thd_mutex_unlock(&mutex);

	return 0;
}

/* Initialize service ABI */
#include <kallisti/svcmpx.h>
#include <kallisti/abi/fs.h>
static abi_fs_t fsabi;
static void fs_abi_init() {
	memset(&fsabi, 0, sizeof(fsabi));

	fsabi.hdr.version = ABI_MAKE_VER(1, 0, 0);
	
	fsabi.open = fs_open;
	fsabi.close = fs_close;
	fsabi.read = fs_read;
	fsabi.write = fs_write;
	fsabi.seek = fs_seek;
	fsabi.tell = fs_tell;
	fsabi.total = fs_total;
	fsabi.readdir = fs_readdir;
	fsabi.ioctl = fs_ioctl;
	fsabi.rename = fs_rename;
	fsabi.unlink = fs_unlink;
	
	fsabi.handler_add = fs_handler_add;
	fsabi.handler_remove = fs_handler_remove;
	
	svcmpx_add_handler("fs", &fsabi);
}

/* Initialize FS structures */
int fs_init() {
	int rv = 0;

	/* Start with no handlers */
	fs_handlers = NULL;

	/* Init thread mutex */
	thd_mutex_reset(&mutex);

	/* Initialize service ABI */
	fs_abi_init();

	return rv;
}
