/* KallistiOS 0.6

   fs_serconsole.c
   (c)2000 Dan Potter
   
*/

static char id[] = "KOS $Id: fs_serconsole.c,v 1.3 2000/11/12 01:04:08 bard Exp $";

/*

fs_serconsole is a VFS handler that interfaces to a program running a PC
over the DC external serial port to implement a host-based filesystem.
Whatever directory tree is exported on the PC will show up in its entirety
on the KOS tree where this is mounted (by default, /pc). Additionally, all
kernel console messages are redirected to the PC host.

No provision is currently in place for reading keyboard input from the
"console" in KOS, but if/when that is implemented this will also handle
that function.

*/

#include <malloc.h>
#include <kallisti/fs/serconsole.h>
#include <kallisti/thread.h>
#include <kallisti/process.h>
#include <kallisti/serial.h>

/******************************************************** Serial interface ****/
/* Access mutex (can only do one thing with the serial port at once) */
static thd_mutex_t mutex;

/* Available console commands */
#define CMD_HELLO	0x48454c4f	/* HELO */
#define CMD_FIRST	1
#define CMD_PRINTK	1
#define CMD_DONE	2
#define CMD_OPENDIR	3
#define CMD_READDIR	4
#define CMD_CLOSEDIR	5
#define CMD_OPENFILE	6
#define CMD_CLOSEFILE	7
#define CMD_READFILE	8
#define CMD_TELLFILE	9
#define CMD_TOTALFILE	10
#define CMD_WRITEFILE	11
#define CMD_LAST	11

/* Available console responses */
#define RESP_OK		0
#define RESP_ERROR	0xffffffff

/* Represents one command packet */
typedef struct {
	uint32	command;
	uint32	datasize;
	uint32	xorsum;
	void	*data;
} packet_t;

/* Block until a char is available */
static int read_block() {
	int c;
	while ( (c=serbuf_read()) < 0)
		thd_pass();
	return c;
}

/* Block until a char is available, or timeout */
static int read_block_to(int32 ms) {
	int c;
	
	while ( (c=serbuf_read()) < 0 && ms > 0) {
		timer_sleep(10);
		ms-=10;
	}
	return c;
}

/* Write a uint32 */
static void write_uint32(uint32 v) {
	serial_write((v >> 24) & 0xff);
	serial_write((v >> 16) & 0xff);
	serial_write((v >> 8) & 0xff);
	serial_write((v >> 0) & 0xff);
	serial_flush();
}

/* Read a uint32 (block until complete) */
static uint32 read_uint32() {
	uint32 v;
	
	v  = read_block() << 24;
	v |= read_block() << 16;
	v |= read_block() << 8;
	v |= read_block() << 0;
	
	return v;
}

/* Calculates the xorsum of a packet */
static uint32 xorsum_data(void *data, int size) {
	uint32 *d = (uint32*)data;
	uint32 sum = 0;
	int i;
	
	if (size % 4) size += 4 - (size % 4);
	
	for (i=0; i<size; i++)
		sum ^= d[i];
	
	return sum;
}

/* Writes a complete command packet to the host software */
static int write_packet(packet_t *p) {
	uint32 ack;
	
	write_uint32(p->command);
	ack = read_uint32();
	if (ack != RESP_OK) return -1;

	/* Crashes sometimes =| */
	/*p->xorsum = xorsum_data(p->data, p->datasize);*/

	write_uint32(p->datasize);
	write_uint32(p->xorsum);
	serial_write_buffer((uint8*)p->data, p->datasize);
	
	ack = read_uint32();
	if (ack != RESP_OK) return -1;
	
	return 0;
}

/* Reads a complete command packet from the host software */
static int read_packet(packet_t *p, int maxsize) {
	p->command = read_uint32();
	write_uint32(RESP_OK);
	
	p->datasize = read_uint32();
	p->xorsum = read_uint32();
	if (p->datasize > maxsize) {
		serbuf_read_buffer((uint8*)p->data, maxsize);
		while (serbuf_read() != -1)
			;
	}
	else {
		serbuf_read_buffer((uint8*)p->data, p->datasize);
	}
	write_uint32(RESP_OK);
	
	return 0;
}

/* Block until the PC has acknowledged us */
static void fscc_link_to_pc() {
	uint32 ack;
	
	while(1) {
		write_uint32(CMD_HELLO);
		ack = read_block_to(1000);
		if (ack == 'O') break;
	}
}

/* Unlink from the PC */
static void fscc_unlink_from_pc() {
	packet_t pkt = { CMD_DONE, 0, 0, NULL };
	write_packet(&pkt);
}

/******************************************************** FS interface ********/


/* Printk replacement */
static void (*old_printk)(char *str) = NULL;
void fscc_printk(char *str) {
	packet_t pkt = { CMD_PRINTK, strlen(str)+1, 0, str };
	thd_mutex_lock(&mutex);
	write_packet(&pkt);
	thd_mutex_unlock(&mutex);
}

uint32 fscc_open(const char *fn, int mode) {
	uint32 hnd = 0;
	
	thd_mutex_lock(&mutex);
	if (mode & O_DIR) {
		packet_t pkt = { CMD_OPENDIR, strlen(fn)+1, 0, fn };

		if (write_packet(&pkt) >= 0) {
			pkt.data = &hnd;
			read_packet(&pkt, 4);
		}
	} else {
		char *buffer = malloc(4+strlen(fn)+1);
		packet_t pkt = { CMD_OPENFILE, 4+strlen(fn)+1, 0, buffer };
		
		*((uint32*)buffer) = mode;
		memcpy(buffer+4, fn, strlen(fn)+1);
		if (write_packet(&pkt) >= 0) {
			pkt.data = &hnd;
			read_packet(&pkt, 4);
		}
	}
	
	thd_mutex_unlock(&mutex);
	return hnd;
}

void fscc_close(uint32 hnd) {
	thd_mutex_lock(&mutex);

	/* Cheap hack */
	if (hnd > 100) {
		packet_t pkt = { CMD_CLOSEDIR, 4, 0, &hnd };
		write_packet(&pkt);
	} else {
		packet_t pkt = { CMD_CLOSEFILE, 4, 0, &hnd };
		write_packet(&pkt);
	}
	thd_mutex_unlock(&mutex);
}

ssize_t fscc_read(uint32 hnd, void *buf, size_t cnt) {
	uint32 data[] = { hnd, cnt };
	packet_t pkt = { CMD_READFILE, 8, 0, data };

	thd_mutex_lock(&mutex);
	if (write_packet(&pkt) < 0) {
		pkt.datasize = -1;
	} else {
		pkt.data = buf;
		if (read_packet(&pkt, cnt) < 0) {
			pkt.datasize = -1;
		}
	}
	
	thd_mutex_unlock(&mutex);
	return pkt.datasize;
}

ssize_t fscc_write(uint32 hnd, void *buf, size_t cnt) {
	uint32 *data;
	packet_t pkt = { CMD_WRITEFILE, 4+4+cnt, 0, NULL };
	
	thd_mutex_lock(&mutex);
	data = malloc(4+4+cnt);
	data[0] = hnd;
	data[1] = cnt;
	memcpy(data+2, buf, cnt);
	
	pkt.data = data;
	
	if (write_packet(&pkt) < 0) {
		cnt = -1;
	} else {
		if (read_packet(&pkt, 4) < 0) {
			cnt = -1;
		} else {
			cnt = data[0];
		}
	}
	
	free(data);
	thd_mutex_unlock(&mutex);
	return cnt;
}

off_t fscc_seek(uint32 hnd, off_t offset, int whence) {
	return -1;
}

off_t fscc_tell(uint32 hnd) {
	return -1;
}

size_t fscc_total(uint32 hnd) {
	return -1;
}

/* Not thread-safe, but that's ok because neither is the FS */
static dirent_t dirent;
dirent_t *fscc_readdir(uint32 hnd) {
	if (hnd < 100) return NULL;
	
{
	packet_t pkt = { CMD_READDIR, 4, 0, &hnd };
	dirent_t *rv = NULL;

	thd_mutex_lock(&mutex);
	if (write_packet(&pkt) >= 0) {
		pkt.data = &dirent;
		if (read_packet(&pkt, sizeof(dirent)) >= 0) {
			rv = &dirent;
		}
	}

	thd_mutex_unlock(&mutex);
	return rv;
}
}

/* Pull all that together */
static vfs_handler vh = {
	0, 0,		/* In-kernel, no cacheing */
	fscc_open,
	fscc_close,
	fscc_read,
	fscc_write,
	fscc_seek,
	fscc_tell,
	fscc_total,
	fscc_readdir,
	NULL,
	NULL,
	NULL
};

/* This is called before any main() gets to take a whack */
int fs_serconsole_init() {
	int i;

	/* Init thread mutex */
	thd_mutex_reset(&mutex);

	/* Interface with the PC program before continuing */
	fscc_link_to_pc();

	/* Set up printk replacement */
	old_printk = ps_set_printk(fscc_printk);

	/* Register with VFS */
	return fs_handler_add("/pc", &vh);
}

int fs_serconsole_shutdown() {
	fscc_unlink_from_pc();
	ps_set_printk(old_printk);
	return fs_handler_remove(&vh);
}

