/* KallistiOS 0.6

   irq.c
   (c)2000 Dan Potter
*/

static char id[] = "KOS $Id: irq.c,v 1.4 2000/11/11 08:40:18 bard Exp $";

/* This module contains low-level handling for IRQs and related exceptions. */

#include <string.h>
#include <stdio.h>
#include <kallisti/stdtypes.h>
#include <kallisti/irq.h>
#include <kallisti/timer.h>

/* Location to store saved CPU registers. We need REG_BYTE_CNT bytes. This space
   is alwaus used unless you want to pursue threading in another module. */
uint32 save_regs_tbl[REG_BYTE_CNT / 4];

/* Exception table -- this table matches (EXPEVT>>4) to a function pointer.
   If the pointer is null, then nothing happens. Otherwise, the function will
   handle the exception. */
static exc_handler exc_handlers[0x100];

/* Global exception handler -- hook this if you want to get each and every
   exception; you might get more than you bargained for, but it can be useful. */
static exc_handler exc_hnd_global;

/* Set a handler, or remove a handler */
int irq_set_handler(uint32 code, exc_handler hnd) {
	/* Make sure they don't do something crackheaded */
	if (code >= 0x1000 || (code & 0x000f)) return -1;
	
	code = code >> 4;
	exc_handlers[code] = hnd;
	
	return 0;
}

/* Set a global handler */
int irq_set_global_handler(exc_handler hnd) {
	exc_hnd_global = hnd;
	return 0;
}

/* The C-level routine that processes context switching and other
   types of interrupts. NOTE: We are running on the stack of the process
   that was interrupted! */
volatile uint32 jiffies = 0;
extern void *srt_addr;
static void handle_exception(int code) {
	/* Get the exception code */
	volatile uint32 *tra = (uint32*)0xff000020;
	volatile uint32 *expevt = (uint32*)0xff000024;
	volatile uint32 *intevt = (uint32*)0xff000028;
	uint32 evt = 0;
	int handled = 0;

	/* If it's a code 0, well, we shouldn't be here. */
	if (code == 0) return;
	
	/* If it's a code 1 or 2, grab the event from expevt. */
	if (code == 1 || code == 2) evt = *expevt;
	
	/* If it's a code 3, grab the event from intevt. */
	if (code == 3) evt = *intevt;

	/* If there's a global handler, call it */
	if (exc_hnd_global) {
		exc_hnd_global(code, evt);
		handled = 1;
	}

	/* printf("got int %04x %04x\r\n", code, evt); */

	/* If it's a timer interrupt, clear the status; if it's timer 0,
		increase the jiffies count */
	if (evt >= EXC_TMU0_TUNI0 && evt <= EXC_TMU2_TUNI2) {
		if (evt == EXC_TMU0_TUNI0) {
			timer_clear(TMU0);
			jiffies++;
		} else if (evt == EXC_TMU1_TUNI1) {
			timer_clear(TMU1);
		} else {
			timer_clear(TMU2);
		}
		handled = 1;
	}

	/* If it's a trapa #0, then pretend it was a timer event */
	if (evt == EXC_TRAPA && (*tra & (0xff<<2)) == 0) {
		code = 3;
		evt = EXC_TMU0_TUNI0;
	}
	
	/* If there's a handler, call it */
	{
		exc_handler hnd = exc_handlers[evt >> 4];
		if (hnd == NULL) return;
		hnd(code, evt);
		handled = 1;
	}
	
	if (!handled) {
		uint32 *regs = (uint32*)srt_addr;
		
		printf("Unhandled exception: PC %08lx, code %d, evt %04x\r\n",
			regs[16], code, (uint16)evt);
		printf(" R0-R7: %08lx %08lx %08lx %08lx %08lx %08lx %08lx %08lx\r\n",
			regs[0], regs[1], regs[2], regs[3], regs[4], regs[5], regs[6], regs[7]);
		printf(" R8-R15: %08lx %08lx %08lx %08lx %08lx %08lx %08lx %08lx\r\n",
			regs[8], regs[9], regs[10], regs[11], regs[12], regs[13], regs[14], regs[15]);
		printf(" SR %08lx PR %08lx\r\n", regs[22], regs[17]);
		printf(" Vicinity code ");
		printf(" @%08lx: %04x %04x %04x %04x %04x\r\n",
			regs[16]-4, *((uint16*)(regs[16]-4)), *((uint16*)(regs[16]-2)),
			*((uint16*)(regs[16])), *((uint16*)(regs[16]+2)), *((uint16*)(regs[16]+4)));
		printf("Rebooting...\r\n");
		{ void (*rb)() = (void (*)())0xa0000000; rb(); }
	}
}

/* Routine that all exception handlers jump to after setting
   an error code. Out of neccessity, all of this function is in
   assembler instead of C. Once the registers are saved, we will
   jump into a shared routine. This register save and restore code
   is mostly from my sh-stub code. For now this is pretty high overhead
   for a context switcher (or especially a timer! =) but it can
   be optimized later. */
asm("
	.text
	.align		2
	.globl		_srt_addr

_save_regs:
! On the SH4, an exception triggers a toggle of RB in SR. So all
! the R0-R7 registers were convienently saved for us.
	mov.l		_srt_addr,r0	! Grab the location of the reg store
	add		#0x20,r0	! Start at the top of the BANK regs
	stc.l		r7_bank,@-r0	! Save R7
	stc.l		r6_bank,@-r0	! Save R6
	stc.l		r5_bank,@-r0	! Save R5
	stc.l		r4_bank,@-r0	! Save R4
	stc.l		r3_bank,@-r0	! Save R3
	stc.l		r2_bank,@-r0	! Save R2
	stc.l		r1_bank,@-r0	! Save R1
	stc.l		r0_bank,@-r0	! Save R0
	
	mov.l		r8,@(0x20,r0)	! save R8
	mov.l		r9,@(0x24,r0)	! save R9
	mov.l		r10,@(0x28,r0)	! save R10
	mov.l		r11,@(0x2c,r0)	! save R11
	mov.l		r12,@(0x30,r0)	! save R12
	mov.l		r13,@(0x34,r0)	! save R13
	mov.l		r14,@(0x38,r0)	! save R14
	mov.l		r15,@(0x3c,r0)	! save R15 (SP)
	add		#0x5c,r0	! readjust register pointer
	stc.l		ssr,@-r0	! save SSR  0x58
	sts.l		macl,@-r0	! save MACL 0x54
	sts.l		mach,@-r0	! save MACH 0x50
	stc.l		vbr,@-r0	! save VBR  0x4c
	stc.l		gbr,@-r0	! save GBR  0x48
	sts.l		pr,@-r0		! save PR   0x44
	stc.l		spc,@-r0	! save PC   0x40
	
	add		#0x60,r0	! readjust register pointer
	add		#0x44,r0
	sts.l		fpul,@-r0	! save FPUL  0xe0
	sts.l		fpscr,@-r0	! save FPSCR 0xdc
	mov		#0,r2		! Set known FP flags
	lds		r2,fpscr
	fmov.s		fr15,@-r0	! save FR15  0xd8
	fmov.s		fr14,@-r0	! save FR14
	fmov.s		fr13,@-r0	! save FR13
	fmov.s		fr12,@-r0	! save FR12
	fmov.s		fr11,@-r0	! save FR11
	fmov.s		fr10,@-r0	! save FR10
	fmov.s		fr9,@-r0	! save FR9
	fmov.s		fr8,@-r0	! save FR8
	fmov.s		fr7,@-r0	! save FR7
	fmov.s		fr6,@-r0	! save FR6
	fmov.s		fr5,@-r0	! save FR5
	fmov.s		fr4,@-r0	! save FR4
	fmov.s		fr3,@-r0	! save FR3
	fmov.s		fr2,@-r0	! save FR2
	fmov.s		fr1,@-r0	! save FR1
	fmov.s		fr0,@-r0	! save FR0   0x9c
	frchg				! Second FP bank
	fmov.s		fr15,@-r0	! save FR15  0x98
	fmov.s		fr14,@-r0	! save FR14
	fmov.s		fr13,@-r0	! save FR13
	fmov.s		fr12,@-r0	! save FR12
	fmov.s		fr11,@-r0	! save FR11
	fmov.s		fr10,@-r0	! save FR10
	fmov.s		fr9,@-r0	! save FR9
	fmov.s		fr8,@-r0	! save FR8
	fmov.s		fr7,@-r0	! save FR7
	fmov.s		fr6,@-r0	! save FR6
	fmov.s		fr5,@-r0	! save FR5
	fmov.s		fr4,@-r0	! save FR4
	fmov.s		fr3,@-r0	! save FR3
	fmov.s		fr2,@-r0	! save FR2
	fmov.s		fr1,@-r0	! save FR1
	fmov.s		fr0,@-r0	! save FR0   0x5c
	frchg				! First FP bank again

	! R4 still contains the exception code
	mov.l		hdl_except,r2	! Call handle_exception
	jsr		@r2
	nop

! Now restore all the registers and jump back to the thread
	mov.l	_srt_addr, r1		! Get register store address
	ldc.l	@r1+,r0_bank		! restore R0
	ldc.l	@r1+,r1_bank		! restore R1
	ldc.l	@r1+,r2_bank		! restore R2
	ldc.l	@r1+,r3_bank		! restore R3
	ldc.l	@r1+,r4_bank		! restore R4
	ldc.l	@r1+,r5_bank		! restore R5
	ldc.l	@r1+,r6_bank		! restore R6
	ldc.l	@r1+,r7_bank		! restore R7
	add	#-32,r1			! Go back to the front
	mov.l	@(0x20,r1), r8		! restore R8
	mov.l	@(0x24,r1), r9		! restore R9
	mov.l	@(0x28,r1), r10		! restore R10
	mov.l	@(0x2c,r1), r11		! restore R11
	mov.l	@(0x30,r1), r12		! restore R12
	mov.l	@(0x34,r1), r13		! restore R13
	mov.l	@(0x38,r1), r14		! restore R14
	mov.l	@(0x3c,r1), r15		! restore programs stack
	
	add	#0x40,r1		! jump up to status words
	ldc.l	@r1+,spc		! restore SPC 0x40
	lds.l	@r1+,pr			! restore PR  0x44
	ldc.l	@r1+,gbr		! restore GBR 0x48
!	ldc.l	@r1+,vbr		! restore VBR (don't play with VBR)
	add	#4,r1
	lds.l	@r1+,mach		! restore MACH 0x50
	lds.l	@r1+,macl		! restore MACL 0x54
	ldc.l	@r1+,ssr		! restore SSR  0x58

	mov	#0,r2			! Set known FP flags
	lds	r2,fpscr
	frchg				! Second FP bank
	fmov.s	@r1+,fr0		! restore FR0  0x5c
	fmov.s	@r1+,fr1		! restore FR1
	fmov.s	@r1+,fr2		! restore FR2
	fmov.s	@r1+,fr3		! restore FR3
	fmov.s	@r1+,fr4		! restore FR4
	fmov.s	@r1+,fr5		! restore FR5
	fmov.s	@r1+,fr6		! restore FR6
	fmov.s	@r1+,fr7		! restore FR7
	fmov.s	@r1+,fr8		! restore FR8
	fmov.s	@r1+,fr9		! restore FR9
	fmov.s	@r1+,fr10		! restore FR10
	fmov.s	@r1+,fr11		! restore FR11
	fmov.s	@r1+,fr12		! restore FR12
	fmov.s	@r1+,fr13		! restore FR13
	fmov.s	@r1+,fr14		! restore FR14
	fmov.s	@r1+,fr15		! restore FR15 0x98
	frchg				! First FP bank
	fmov.s	@r1+,fr0		! restore FR0  0x9c
	fmov.s	@r1+,fr1		! restore FR1
	fmov.s	@r1+,fr2		! restore FR2
	fmov.s	@r1+,fr3		! restore FR3
	fmov.s	@r1+,fr4		! restore FR4
	fmov.s	@r1+,fr5		! restore FR5
	fmov.s	@r1+,fr6		! restore FR6
	fmov.s	@r1+,fr7		! restore FR7
	fmov.s	@r1+,fr8		! restore FR8
	fmov.s	@r1+,fr9		! restore FR9
	fmov.s	@r1+,fr10		! restore FR10
	fmov.s	@r1+,fr11		! restore FR11
	fmov.s	@r1+,fr12		! restore FR12
	fmov.s	@r1+,fr13		! restore FR13
	fmov.s	@r1+,fr14		! restore FR14
	fmov.s	@r1+,fr15		! restore FR15  0xd8
	lds.l	@r1+,fpscr		! restore FPSCR 0xdc
	lds.l	@r1+,fpul		! restore FPUL  0xe0

	add	#-0x70,r1		! jump back to registers
	add	#-0x34,r1
	mov.l	@(0,r1),r0		! restore R0
	mov.l	@(4,r1),r1		! restore R1

	rte				! return
	nop
	
	.align 4
_srt_addr:
	.long	0	! Save Regs Table -- this is an indirection
			! so we can easily swap out pointers during a
			! context switch.
hdl_except:
	.long	_handle_exception + 0x20000000
");

/* The SH4 has very odd exception handling. Instead of having a vector
   table like a sensible processor, it has a vector code block. *sigh*
   Thus this table of assembly code. Note that we can't catch reset
   exceptions at all, but that really shouldn't matter. */
extern void vma_table();
asm("
	.text
	.align 4
_vma_table:
	.rep	0x100/4
	.long	0
	.endr
	
_vma_table_100:		! General exceptions
	mov.l	sr_100,r4		! Get jump location
	jmp	@r4
	mov	#1,r4			! Set exception code
	nop
sr_100:
	.long	_save_regs
	
	.rep	(0x300/4) - (12/4)
	.long	0
	.endr

_vma_table_400:		! TLB miss exceptions (MMU)
	mov.l	sr_400,r4		! Get jump location
	jmp	@r4
	mov	#2,r4			! Set exception code
	nop
sr_400:
	.long	_save_regs

	.rep	(0x200/4) - (12/4)
	.long	0
	.endr

_vma_table_600:		! IRQs
	mov.l	sr_600,r4		! Get jump location
	jmp	@r4
	mov	#3,r4			! Set exception code
	nop
sr_600:
	.long	_save_regs
");


/* Switches register banks; call this outside of exception handling
   (but make sure interrupts are off!!) to change where registers will
   go to, or call it inside an exception handler to switch contexts. 
   Make sure you have at least REG_BYTE_CNT bytes available. */
void irq_change_context(void *regbank) {
	if (regbank == NULL)
		srt_addr = save_regs_tbl;
	else
		srt_addr = regbank;
}

/* Disable interrupts: DO NOT CALL INSIDE AN EXCEPTION HANDLER */
void irq_disable() {asm("

	stc	sr,r0
	mov.l	_irqd_or,r1
	or	r1,r0
	ldc	r0,sr
	bra	_irqd_past_data
	
	.align 4
_irqd_or:
	.long	0x100000f0

_irqd_past_data:

");}

/* Enable interrupts: DO NOT CALL INSIDE AN EXCEPTION HANDLER */
void irq_enable() {asm("

	stc	sr,r0
	mov.l	_irqe_and,r1
	and	r1,r0
	ldc	r0,sr
	bra	_irqe_past_data
	
	.align 4
_irqe_and:
	.long	0xcfffff0f

_irqe_past_data:

");}

/* Retrieve SR */
uint32 irq_get_sr() {
	asm("stc	sr,r0");
}

/* Pre-init SR and VBR */
static uint32 pre_sr, pre_vbr;

/* Init routine */
void irq_init() {
	/* Save SR and VBR */
	asm("stc	sr,r0
	    mov.l	r0,%0" : : "m"(pre_sr));
	asm("stc	vbr,r0
	    mov.l	r0,%0" : : "m"(pre_vbr));

	/* Setup a single register store table for now */
	irq_change_context(save_regs_tbl);

	/* Blank the exception handler table */
	memset(exc_handlers, 0, sizeof(exc_handlers));
	exc_hnd_global = NULL;

	/* Set VBR to our exception table above, and enable exceptions and IRQs */
	asm("
		! Set VBR
		mov.l	_vbr_addr,r0
		ldc	r0,vbr
		
		! Set SR (RB=0, BL=0, IL=0)
		stc	sr,r0
		mov.l	_sr_and,r1
		and	r1,r0
		ldc	r0,sr
		
		bra	_after_vbr
		nop
		.align 4
	_vbr_addr:
		.long	_vma_table		! Non-cached area
	_sr_and:
		.long	0xcfffff0f
	_after_vbr:
	");
}


void irq_shutdown() {
	/* Restore SR and VBR */
	asm("mov.l	%0,r0
	    ldc	r0,sr" : : "m"(pre_sr));
	asm("mov.l	%0,r0
	    ldc	r0,vbr" : : "m"(pre_vbr));

}





