/* KallistiOS 0.6
 *
 * maple.c
 * (c)2000 Jordan DeLong
 */

static char id[] = "KOS $Id: maple.c,v 1.3 2000/11/11 06:11:48 fracture Exp $";

#include <kallisti/maple.h>
#include <kallisti/thread.h>
#include <malloc.h>
#include <string.h>
#include <stdio.h>

/*
 * This module handles low-level communication/initialization of the maple 
 * bus.  Specific devices aren't handled by this module, rather, the modules
 * implementing specific devices can use this module to access them.
 * 
 * Thanks to Marcus Comstedt for information on the maple bus.
 */

/* mutex for access to this module */
static thd_mutex_t mutex;

/* macro for poking maple mem */
#define MAPLE(x) (*(unsigned long *)((0xa05f6c00)+(x)))

/* dma transfer buffer */
uint32 *dmabuffer;

/* a few small internal functions to make code slightly more readable */
void maple_enable_bus() { MAPLE(0x14) = 1; }
void maple_disable_bus() { MAPLE(0x14) = 0; }
void maple_start_dma() { MAPLE(0x18) = 1; }
int maple_dma_in_progress() { return MAPLE(0x18) & 1; }
/* set timeout to timeout, and bitrate to 2Mbps (what it must be) */
void maple_set_timeout(int timeout) { MAPLE(0x80) = (timeout << 16) | 0; }

/* set the DMA ptr */
void maple_set_dma_ptr(unsigned long *ptr) { 
	MAPLE(0x04) = ((uint32) ptr) & 0xfffffff;
}

/* initialize the maple bus. */
static void maple_svc_init();
void maple_init() {
	/* reset hardware */
	MAPLE(0x8c) = 0x6155404f;
	MAPLE(0x10) = 0;

	maple_set_timeout(50000);

	/* buffer for dma transfers; room for a send and recv buffer */
	dmabuffer = memalign(32, 1024 + 1024 + 4 + 4);

	maple_enable_bus();
	
	maple_rescan_bus(0);
	
	maple_svc_init();
}

/* turn off the maple bus, free mem */
void maple_shutdown() {
	svcmpx_remove_handler("maple");

	free(dmabuffer);
	maple_disable_bus();
}

/* use the information in tdesc to place a new transfer descriptor, followed by
   a num of frames onto buffer, and return the new ptr into buff */
uint32 *maple_add_trans(maple_tdesc_t tdesc, uint32 *buffer) {
	int i;
	
	/* build the transfer descriptor's first word */
	*buffer++ = tdesc.length | (tdesc.port << 16) | (tdesc.lastdesc << 31);
	/* transfer descriptor second word: address to receive buffer */
	*buffer++ = ((uint32) tdesc.recvaddr) & 0xfffffff;
	
	/* add each of the frames in this transfer */
	for (i = 0; i < tdesc.numframes; i++) {
		/* frame header */
		*buffer++ = (tdesc.frames[i].cmd & 0xff) | (tdesc.frames[i].to << 8)
			| (tdesc.frames[i].from << 16) | (tdesc.frames[i].datalen << 24);
		/* parameter data, if any exists */
		if (tdesc.frames[i].datalen > 0) {
			memcpy(buffer, tdesc.frames[i].data, tdesc.frames[i].datalen * 4);
			buffer += tdesc.frames[i].datalen;
		}
	}
	
	return buffer;
}

/* read information at buffer and turn it into a maple_frame_t */
void maple_read_frame(uint32 *buffer, maple_frame_t *frame) {
	uint8 *b = (uint8 *) buffer;

	frame->cmd = b[0];
	frame->to = b[1];
	frame->from = b[2];
	frame->datalen = b[3];
	frame->data = &b[4];
}

/* create a maple address, -1 on error */
uint8 maple_create_addr(uint8 port, uint8 unit) {
	uint8 addr;
	
	if (port > 3 || unit > 5) return -1;
	
	addr = port << 6;
	if (unit != 0)
		addr |= (1 << (unit - 1)) & 0x1f;
	else
		addr |= 0x20;
		
	return addr;
}

/* initiate a dma transfer and block until it's completed,
   return -1 if error (currently only if DMA is already in
   progress) */
int maple_dodma_block() {
	if (maple_dma_in_progress())
		return -1;

	/* enable maple dma transfer bit */
	maple_start_dma();
	
	/* wait for it to clear (indicating the end of the transfer */
	while (maple_dma_in_progress())
		;
	
	return 0;
}

/* little funct to send a single command on the maple bus, and block until 
   recving a response, returns -1 on error */
int maple_docmd_block(int8 cmd, uint8 addr, uint8 datalen, void *data, maple_frame_t *retframe) {
	uint32 *sendbuff, *recvbuff;
	maple_tdesc_t tdesc;
	maple_frame_t frame;

	thd_mutex_lock(&mutex);

	/* setup buffer ptrs, into dmabuffer and set uncacheable */
	sendbuff = (uint32 *) dmabuffer;
	recvbuff = (uint32 *) (dmabuffer + 1024);
	sendbuff = (uint32 *) ((uint32) sendbuff | 0xa0000000);
	recvbuff = (uint32 *) ((uint32) recvbuff | 0xa0000000);

	/* setup tdesc/frame */
	tdesc.lastdesc = 1;
	tdesc.port = addr >> 6;
	tdesc.length = datalen;
	tdesc.recvaddr = recvbuff;
	tdesc.numframes = 1;
	tdesc.frames = &frame;
	frame.cmd = cmd;
	frame.data = data;
	frame.datalen = datalen;
	frame.from = tdesc.port << 6;
	frame.to = addr;

	/* make sure no DMA is in progress */
	if (maple_dma_in_progress())
		return -1;
		
	maple_set_dma_ptr(sendbuff);

	maple_add_trans(tdesc, sendbuff);
					  
	/* do the dma, blocking till it finishes */	
	if (maple_dodma_block() == -1)
		return -1;
	
	maple_read_frame(recvbuff, retframe);
	thd_mutex_unlock(&mutex);
	return 0;
}


/* Rescan the maple bus to recognize VMUs, etc */
static int func_codes[4][6] = { {0} };
int maple_rescan_bus(int quiet) {
	int port, unit, to;
	maple_frame_t frame;

	if (!quiet)
		printf("Rescanning maple bus:\r\n");	
	for (port=0; port<4; port++)
		for (unit=0; unit<6; unit++) {
			to = 1000;
			do {
				if (maple_docmd_block(MAPLE_COMMAND_DEVINFO,
						maple_create_addr(port, unit), 0, NULL, &frame)
						== -1)
					return -1;
				to--;
				if (to <= 0) {
					printf("  %c%c: timeout\r\n", 'a'+port, '0'+unit);
					break;
				}
			} while (frame.cmd == MAPLE_RESPONSE_AGAIN);
			if (frame.cmd == MAPLE_RESPONSE_DEVINFO) {
				maple_devinfo_t *di = (maple_devinfo_t*)frame.data;
				di->product_name[29] = '\0';
				func_codes[port][unit] = di->func;
				if (!quiet)
					printf("  %c%c: %s (%08lx)\r\n", 'a'+port, '0'+unit, di->product_name, di->func);
			} else
				func_codes[port][unit] = 0;
		}
		
	return 0;
}


/* A couple of convienence functions to load maple peripherals */

/* Retrieve function code... */
uint32 maple_device_func(int port, int unit) {
	return func_codes[port][unit];
}

/* First with a given function code... */
uint8 maple_first_device(int code) {
	int port, unit;

	for (port=0; port<4; port++)
		for (unit=0; unit<6; unit++) {
			if (func_codes[port][unit] & code)
				return maple_create_addr(port, unit);
		}
	return 0;
}

/* First controller */
uint8 maple_first_controller() {
	return maple_first_device(MAPLE_FUNC_CONTROLLER);
}

/* First mouse */
uint8 maple_first_mouse() {
	return maple_first_device(MAPLE_FUNC_MOUSE);
}

/* First keyboard */
uint8 maple_first_kb() {
	return maple_first_device(MAPLE_FUNC_KEYBOARD);
}

/* First LCD unit */
uint8 maple_first_lcd() {
	return maple_first_device(MAPLE_FUNC_LCD);
}

/* First VMU */
uint8 maple_first_vmu() {
	return maple_first_device(MAPLE_FUNC_MEMCARD);
}


/* Setup a service */
#include <kallisti/svcmpx.h>
#include <kallisti/abi/maple.h>
static abi_maple_t mabi;
static void maple_svc_init() {
	memset(&mabi, 0, sizeof(mabi));
	
	mabi.hdr.version = ABI_MAKE_VER(1,0,0);

	mabi.create_addr = maple_create_addr;
	mabi.docmd_block = maple_docmd_block;
	mabi.rescan_bus = maple_rescan_bus;
	mabi.device_func = maple_device_func;
	mabi.first_device = maple_first_device;
	mabi.first_controller = maple_first_controller;
	mabi.first_mouse = maple_first_mouse;
	mabi.first_kb = maple_first_kb;
	mabi.first_lcd = maple_first_lcd;
	mabi.first_vmu = maple_first_vmu;
	
	mabi.cont_get_cond = cont_get_cond;
	
	mabi.kbd_get_cond = kbd_get_cond;
	mabi.kbd_set_queue = kbd_set_queue;
	mabi.kbd_enqueue = kbd_enqueue;
	mabi.kbd_get_key = kbd_get_key;
	mabi.kbd_poll = kbd_poll;
	
	mabi.mouse_get_cond = mouse_get_cond;
	
	mabi.vmu_draw_lcd = vmu_draw_lcd;
	mabi.vmu_block_read = vmu_block_read;
	mabi.vmu_block_write = vmu_block_write;

	svcmpx_add_handler("maple", &mabi);
}

