/* KallistiOS 0.6
  
   spu.c
   (c)2000 Dan Potter
 */

static char id[] = "KOS $Id: spu.c,v 1.3 2000/11/12 01:14:59 bard Exp $";

#include <kallisti/spu.h>

/*

This module handles the sound processor unit (SPU) of the Dreamcast system.
The processor is a Yamaha AICA, which is powered by an ARM7 RISC core.
To operate the CPU, you simply put it into reset, load a program and
potentially some data into the sound ram, and then let it out of reset. The
ARM will then start executing your code.

In the interests of simplifying the programmer's task, KallistiOS has
made available several default sound programs. One of them is designed to
play MIDI-style tracker data (converted S3M/XM/MOD/MIDI/etc) and the other
is designed to play buffered sound data. Each of these has an associated
API that can be used from the SH-4. Note that the act of referencing
either in your program statically causes them to be linked into the
kernel; so don't use them if you don't need to =).

*/

static volatile uint8 *dc_snd_base = (uint8 *)0xa0700000;

/* Some convienence macros */
#define SNDREG32A(x) ((volatile uint32 *)(dc_snd_base + (x)))
#define SNDREG32(x) (*SNDREG32A(x))
#define SNDREG8A(x) (dc_snd_base + (x))
#define SNDREG8(x) (*SNDREG8A(x))
#define CHNREG32A(chn, x) SNDREG32A(0x80*(chn) + (x))
#define CHNREG32(chn, x) (*CHNREG32A(chn, x))
#define CHNREG8A(chn, x) SNDREG8A(0x80*(chn) + (x))
#define CHNREG8(chn, x) (*CHNREG8A(chn, x))


/* Waits for the sound ram FIFO to empty */
void spu_ram_write_wait() {
	volatile unsigned long *snd_fifo = (unsigned long*)0xa05f688c;
	int i;

	for (i=0; i<0x1800; i++) {
		if (*snd_fifo & 1) break;
	}
}

/* memcpy and memset designed for sound RAM; for addresses, don't
   bother to include the 0xa0800000 offset that is implied. 'length'
   must be a multiple of 4, but if it is not it will be rounded up. */
void spu_memload(uint32 toi, uint8 *from, int length) {
	uint32 *froml = (uint32 *)from;
	uint32 *to = (uint32 *)(0xa0800000 + toi);
	int i;
	
	if (length % 4)
		length = (length/4)+1;
	else
		length = length/4;
	
	for (i=0; i<length; i++) {
		*to++ = *froml++;
		if (i && !(i % 8)) spu_ram_write_wait();
	}
}

void spu_memset(uint32 toi, unsigned long what, int length) {
	uint32 *to = (uint32 *)(0xa0800000 + toi);
	int i;
	
	if (length % 4)
		length = (length/4)+1;
	else
		length = length/4;
	
	for (i=0; i<length; i++) {
		*to++ = what;
		if (i && !(i % 8)) spu_ram_write_wait();
	}
}

/* Enable/disable the SPU; note that disable implies reset of the
   ARM CPU core. */
void spu_enable() {
	SNDREG32(0x2c00) &= ~1;
}

void spu_disable() {
	int i;

	/* Stop the ARM processor */
	SNDREG32(0x2c00) |= 1;

	/* Make sure we didn't leave any notes running */
	for (i=0; i<64; i++) {
		CHNREG32(i, 0) = (CHNREG32(i, 0) & ~0x4000) | 0x8000;
	}
}

/* Setup a service */
#include <kallisti/svcmpx.h>
#include <kallisti/abi/spu.h>
static abi_spu_t wasabi;	/* heh */
static void spu_svc_init() {
	memset(&wasabi, 0, sizeof(wasabi));
	
	wasabi.hdr.version = ABI_MAKE_VER(1,0,0);
	
	wasabi.ram_write_wait = spu_ram_write_wait;
	wasabi.memload = spu_memload;
	wasabi.memset = spu_memset;
	wasabi.enable = spu_enable;
	wasabi.disable = spu_disable;
	
	svcmpx_add_handler("spu", &wasabi);
}


/* Initialize the SPU; by default it will be left in a state of
   reset until you upload a program. */
int spu_init() {
	/* Stop the ARM */
	spu_disable();

	/* Clear out sound RAM */
	spu_memset(0, 0, 0x200000/4);

	/* Setup a service */
	spu_svc_init();

	return 0;
}

/* Shutdown SPU */
int spu_shutdown() {
	svcmpx_remove_handler("spu");
	return spu_init();
}


