/* KallistiOS 0.6

   ta.h
   (c)2000 Dan Potter
   
   $Id: ta.h,v 1.3 2000/11/09 04:41:02 bard Exp $
*/

#ifndef __KALLISTI_TA_H
#define __KALLISTI_TA_H

#include <kallisti/stdtypes.h>

/* Background plane data: this is a sort of truncated normal
   polygon data set. I'm assuming it's stored in internalized
   PVR format. It's a three-vertex polygon strip. */
typedef struct {
	uint32		flags1, flags2;
	uint32		dummy;
	float		x1, y1, z1;
	uint32		argb1;
	float		x2, y2, z2;
	uint32		argb2;
	float		x3, y3, z3;
	uint32		argb3;
} ta_bkg_poly;

/* Global background data structure; this is used during the
   rendering process. */
extern ta_bkg_poly *ta_bkg;


/* Polygon header; each polygon you send to the TA needs to have
   this as its first structure. For flags info, see maiwe's doc. */
typedef struct {
	uint32	flags1, flags2, flags3, flags4;
	uint32	dummy1, dummy2, dummy3, dummy4;
} poly_hdr_t;

/* Vertex structure; each polygon has three or more of these
   arranged in a structure called a strip. If bit 28 is set
   in the flags word, it signifies the end of a strip. Note that
   technically multiple strips don't have to be adjacent but
   they are still part of the same polygon. */

/* Opaque Colored vertex */
typedef struct {
	uint32	flags;
	float	x, y, z;
	float	a, r, g, b;
} vertex_oc_t;

/* Opaque Textured vertex */
typedef struct {
	uint32	flags;
	float	x, y, z, u, v;
	uint32	dummy1, dummy2;
	float	a, r, g, b;
	float	oa, or, og, ob;
} vertex_ot_t;

/* Current page */
extern int ta_curpage;

/* Prepare the TA for page flipped 3D */
void ta_init();

/* Turn off TA -- including turning off pesky interrupts */
void ta_shutdown();

/* Send a store queue full of data to the TA */
void ta_send_queue(void *sql, int size);

/* Begin the rendering process for one frame */
void ta_begin_render();

/* Commit a polygon header to the TA */
void ta_commit_poly_hdr(void *polyhdr);

/* Commit a vertex to the TA; include sizeof() parameter */
void ta_commit_vertex(void *vertex, int size);

/* Commit an end-of-list to the TA */
void ta_commit_eol();

/* Finish rendering a frame; this assumes you have written
   a completed display list to the TA. It sets everything up and
   waits for the next vertical blank period to switch buffers. */
void ta_finish_frame();

/* Build a polygon header from the given parameters */
void ta_poly_hdr_col(poly_hdr_t *target, int translucent);
void ta_poly_hdr_txr(poly_hdr_t *target, int translucent,
	int textureformat, int tw, int th, uint32 textureaddr,
	int filtering);

/* Load texture data into the PVR ram */
void ta_load_texture(uint32 dest, void *src, int size);

/* Return a pointer to write to the texture ram directly */
void *ta_texture_map(uint32 loc);

/* Vertex constants */
#define TA_VERTEX_NORMAL	0xe0000000
#define TA_VERTEX_EOL		0xf0000000

/* Translucency constants */
#define TA_OPAQUE		0
#define TA_TRANSLUCENT		1

/* Texture format constants */
#define TA_NO_TEXTURE		0

#define TA_ARGB1555		((0<<1) | 1)	/* Flat versions */
#define TA_RGB565		((1<<1) | 1)
#define TA_ARGB4444		((2<<1) | 1)
#define TA_YUV422		((3<<1) | 1)
#define TA_BUMP			((4<<1) | 1)

#define TA_ARGB1555_TWID	(0<<1)		/* Twiddled versions */
#define TA_RGB565_TWID		(1<<1)
#define TA_ARGB4444_TWID	(2<<1)
#define TA_YUV422_TWID		(3<<1)
#define TA_BUMP_TWID		(4<<1)

/* Filtering constants */
#define TA_NO_FILTER		0
#define TA_BILINEAR_FILTER	1


#endif	/* __KALLISTI_TA_H */
