/* KallistiOS 0.6

   process.c
   (C)2000 Dan Potter and Jordan DeLong
*/

static char id[] = "KOS $Id: process.c,v 1.5 2000/11/09 04:58:18 bard Exp $";

#include <kallisti/elf.h>
#include <kallisti/fs.h>
#include <kallisti/thread.h>
#include <kallisti/serial.h>

/* The multiplexer routine that retrieves services by name for
   running programs; the default one returns no services. */
void *psm_default(char *svc) { return NULL; }
void* (*ps_multiplexer)(char *svc) = psm_default;

/* The printk routine that will be called when the kernel wants to
   print a console message. */
void (*ps_printk_func)(char *msg) = serial_write_str;

/* Routine to flush parts of cache.. thanks to the Linux-SH guys */
/* This routine was taken from sh-stub.c which had a prominent
   no-copyright notice at the top. Therefore, I'm including it now
   under the copyright umbrella of KallistiOS. */
#define L1_CACHE_BYTES 32
#define CACHE_IC_ADDRESS_ARRAY	0xf0000000
#define CACHE_IC_ENTRY_MASK	0x1fe0
struct __large_struct { uint32 buf[100]; };
#define __m(x) (*(struct __large_struct *)(x))

/* end is really cnt */
static void flush_icache_range(uint32 start, uint32 end) {
	uint32 addr, data, v;

	end += start;
	start &= ~(L1_CACHE_BYTES-1);
	
	for (v=start; v<end; v+=L1_CACHE_BYTES) {
		/* Write back O Cache */
		asm volatile("ocbwb %0"
			: /* no output */
			: "m" (__m(v)));
		/* Invalidate I Cache */
		//addr = CACHE_IC_ADDRESS_ARRAY |
		//	(v & CACHE_IC_ENTRY_MASK) | 0x8	/* A-bit */;
		addr = CACHE_IC_ADDRESS_ARRAY |
			(v & CACHE_IC_ENTRY_MASK);
		data = (v & 0xfffffc00);	/* Valid = 0 */
		*((volatile uint32*)addr) = data;
	}
}

/* Loads and relocates an ELF binary from a VFS file. */
elf_prog_t *ps_load(char *fn) {
	uint32 fd;
	elf_prog_t *rv;

	/* Try to open the file */
	fd = fs_open(fn, O_RDONLY);
	if (fd == 0) {
		printf("Can't open binary file %s\r\n", fn);
		return NULL;
	}

	/* Do the ELF load */
	rv = elf_load(fd);
	if (rv == NULL) {
		printf("Couldn't load ELF file %s\r\n", fn);
		return NULL;
	}

	/* Clear the cache for the loaded area */
	flush_icache_range((uint32)rv->data, rv->size);
	
	return rv;
}

/* Loads an ELF binary from a VFS file and execution of the current
   thread pauses until the binary finishes. If it does not finish,
   execution never returns. */
int ps_load_and_exec(char *fn, int argc, char **argv) {
	elf_prog_t *prog;

	/* Load the binary */
	printf("ps_load_and_exec: loading binary %s\r\n", fn);
	if ((prog = ps_load(fn)) == NULL)
		return -1;
	
	/* Call the entry point */
	printf("Calling entry point %08x\r\n", prog->ko_main);
	prog->ko_main(argc, argv);
	printf("Returned from entry\r\n");
	return 0;
}

/* Thread start wrapper */
void ps_fork_start(void *data) {
	elf_prog_t *prog = (elf_prog_t *)data;
	printf("Forked call to ko_main (%08x)\r\n", prog->ko_main);
	prog->ko_main(0, NULL);
	printf("Forked call to ko_main (%08x) DONE\r\n", prog->ko_main);
	printf("Freeing program memory\r\n");
	elf_free(prog);
	printf("Done\r\n");
}

/* Loads an ELF binary and execution of the program begins in a new
   kernel thread. */
int ps_load_and_fork(char *fn) {
	elf_prog_t *prog;
	
	/* Load the binary */
	if ((prog = ps_load(fn)) == NULL)
		return -1;
		
	/* Call the entry point in a new thread */
	return thd_create(ps_fork_start, prog);
}

/* Sets a multiplexer function target (i.e., what programs will
   receive for their ko_get_svc pointer) */
int ps_set_multiplexer(void* (*func)(char *)) {
	ps_multiplexer = func;
	return 0;
}

/* Sets a kernel console function target (determines where kernel
   output will go from a printf() call). Defaults to serial output. Set
   to NULL to disable all kernel output. */
void * ps_set_printk(void (*print)(char *str)) {
	void *old = ps_printk_func;
	ps_printk_func = print;
	return old;
}

/* Wrapper for printk: call this to send data to the console output
   device (whatever that might be) */
void ps_printk(char *str) {
	if (ps_printk_func != NULL)
		ps_printk_func(str);
}

/* set up the service */
#include <kallisti/svcmpx.h>
#include <kallisti/abi/process.h>
static abi_process_t pabi;
static void process_svc_init() {
	memset(&pabi, 0, sizeof(pabi));

	pabi.hdr.version = ABI_MAKE_VER(1, 0, 0);

	pabi.load = ps_load;
	pabi.load_and_exec = ps_load_and_exec;
	pabi.load_and_fork = ps_load_and_fork;

	svcmpx_add_handler("process", &pabi);
}

/* init */
int ps_init() {
	process_svc_init();

	return 0;
}

/* shutdown */
int ps_shutdown() {
	svcmpx_remove_handler("process");

	return 0;
}

