/* KallistiOS 0.6

   thread.c
   (c)2000 Dan Potter
*/

static char id[] = "KOS $Id: thread.c,v 1.3 2000/11/09 04:58:18 bard Exp $";

#include <malloc.h>
#include <string.h>
#include <kallisti/thread.h>
#include <kallisti/irq.h>
#include <kallisti/timer.h>
#include <kallisti/config.h>

/* Are threads enabled? */
int thd_enabled = 0;

/* Next available thread id */
uint32 thd_next_id;

/* The currently executing thread (of a circular chain) */
kthread_t *thd_current;

/* kthread_t describing the kernel thread (for bootstrapping) */
kthread_t *kernthd;

/* Thread execution wrapper; when the thd_create function below
   adds a new thread to the thread chain, this function is the one
   that gets called in the new context. */
void thd_start(void (*routine)(void *param), void *param) {
	printf("Inside thd_start: routine is %08x, param is %08x\r\n", routine, param);

	/* Call the thread function */
	routine(param);
	
	/* Call Dr. Kevorkian */
	thd_destroy(thd_current->tid);
	
	/* Wait for death */
	while(1)
		;
}

/* New thread function; given a routine address, it will create a
   new kernel thread with a default 4k stack. When the routine
   returns, the thread will exit. Returns the new thread id. */
uint32 thd_create(void (*routine)(void *param), void *param) {
	kthread_t *i;

	/* Create a new thread structure */
	kthread_t *nt = malloc(sizeof(kthread_t));
	if (nt == NULL) return 0;
	
	/* Populate the context from the kernel, but change a few
	   key registers. */
	nt->tid = thd_next_id++;
	nt->flags = 0;
	nt->next_jiffy = 0;
	memcpy(nt->context, &kernthd->context, sizeof(nt->context));
	nt->context[0x16] = 0x40000000;					/* SSR */
	nt->context[0x10] = (uint32)thd_start;				/* SPC */
	nt->context[0x0f] = ((uint32)nt->stack) + sizeof(nt->stack);	/* R15 */
	nt->context[0x04] = (uint32)routine;				/* R4 */
	nt->context[0x05] = (uint32)param;				/* R5 */
	
	/* Add it into the thread chain after the current thread */
	irq_disable();
	nt->next = thd_current->next;
	thd_current->next = nt;
	irq_enable();
	
	return nt->tid;
}

/* Given a thread id, this function removes the thread from
   the execution chain. */
int thd_destroy(uint32 tid) {
	kthread_t *cur = thd_current, *first = thd_current;
	int rv = -1;
	
	irq_disable();
	do {
		if (cur->tid == tid) {
			cur->flags |= THD_DEAD;
			rv = 0;
			break;
		}
		cur = cur->next;
	} while (cur != first);
	irq_enable();
	
	return rv;
}

/* Thread switcher function; just swaps out register banks in a
   timer interrupt. We're called inside an exception handler so
   there's no need to irq_disable()/irq_enable(). */
void thd_switch(uint32 code, uint32 evt) {
	/* This is a mark-and-sweep garbage collector; the threads
	   that are dead have been marked as such by thd_destroy.
	   Now what we do here is move forwards until we find a
	   non-dead thread, killing any others that pop up in the
	   way. */
	kthread_t *last;
	do {
		last = thd_current;
		thd_current = thd_current->next;
		if (thd_current->flags & THD_DEAD) {
			last->next = thd_current->next;
			free(thd_current);
			thd_current = last->next;
		}
	} while ( (thd_current->flags & THD_DEAD)
		|| (thd_current->next_jiffy && (jiffies < thd_current->next_jiffy)) );

	thd_current->next_jiffy = 0;
	/* Switch register contexts */
	/*printf("Switching to context %08x, PC=%08x, R15=%08x, SR=%08x\r\n",
		thd_current, thd_current->context[16], thd_current->context[15],
		thd_current->context[22]); */
	irq_change_context(thd_current->context);
}

/* Execute within a thread's timeslice to voluntarily give up
   the current timeslice. This is probably not the best way to
   do this but it works for now. */
void thd_pass() {
	if (!thd_enabled) return;

	/* The exception handler will be looking for these as
	   a manual time slice */
	asm("trapa	#0");
}

/* Thread blocking based sleeping; this is the preferred way to
   sleep because it eases the load on the system for the other
   threads. */
void thd_sleep_jiffies(uint32 jif) {
	/* This is broken right now for some reason... */
	/*thd_current->next_jiffy = jiffies + jif;
	thd_pass(); */
	timer_sleep_jiffies(jif);
}

void thd_sleep(uint32 ms) {
	thd_sleep_jiffies(ms * HZ / 1000);
}


/* Mutex lock/unlock */
void thd_mutex_reset(volatile thd_mutex_t *mutex) {
	*mutex = 0;
}

void thd_mutex_lock(volatile thd_mutex_t *mutex) {
	if (!thd_enabled) return;
	
	/*printf("Locking mutex %08x (value %d)\r\n", mutex, *mutex);*/
	
	while (*mutex != 0)
		//;
		thd_pass();
	*mutex = 1;
	/*printf("Success locking mutex %08x (value %d)\r\n", mutex, *mutex);*/
}

void thd_mutex_unlock(volatile thd_mutex_t *mutex) {
	/*if (thd_enabled)
		printf("Unlocking mutex %08x (value %d)\r\n", mutex, *mutex);*/
	*mutex = 0;
}

/* Event trigger/wait */
void thd_event_reset(volatile thd_event_t *event) {
	*event = 0;
}

void thd_event_set(volatile thd_event_t *event) {
	*event = 1;
}

void thd_event_wait(volatile thd_event_t *event) {
	if (!thd_enabled) return;

	while (*event == 0)
		thd_pass();
}


/* Enable/disable threading temporarily */
void thd_enable(int old) {
	if (old)
		timer_enable_ints(TMU0);
	thd_enabled = 1;
}

int thd_disable() {
	int rv = timer_ints_enabled(TMU0);
	thd_enabled = 0;
	timer_disable_ints(TMU0);
	return rv;
}


/* Get jiffies count */
uint32 thd_jiffies() {
	return jiffies;
}


/* Setup a service */
#include <kallisti/svcmpx.h>
#include <kallisti/abi/thread.h>
static abi_thread_t tabi;
static void thd_svc_init() {
	memset(&tabi, 0, sizeof(tabi));
	
	tabi.hdr.version = ABI_MAKE_VER(1,0,0);

	tabi.create = thd_create;
	tabi.destroy = thd_destroy;
	tabi.pass = thd_pass;
	tabi.sleep_jiffies = thd_sleep_jiffies;
	tabi.sleep = thd_sleep;
	tabi.mutex_reset = thd_mutex_reset;
	tabi.mutex_lock = thd_mutex_lock;
	tabi.mutex_unlock = thd_mutex_unlock;
	tabi.event_reset = thd_event_reset;
	tabi.event_set = thd_event_set;
	tabi.event_wait = thd_event_wait;
	tabi.jiffies = thd_jiffies;

	svcmpx_add_handler("thread", &tabi);
}


/* Init */
int thd_init() {
	/* Setup the kernel thread */
	kernthd = malloc(sizeof(kthread_t));
	kernthd->tid = 0;
	kernthd->flags = 0;
	kernthd->next_jiffy = 0;
	memset(kernthd->context, 0, sizeof(kernthd->context));
	memset(kernthd->stack, 0, sizeof(kernthd->stack));
	kernthd->next = kernthd;

	/* Start at tid 1 */
	thd_next_id = 1;
	
	/* One thread -- the kernel */
	thd_current = kernthd;

	/* Enable threading */
	irq_disable();
	irq_change_context(kernthd->context);
	irq_set_handler(EXC_TMU0_TUNI0, thd_switch);
	irq_enable();
	
	/* Wait for the first context switch so that the kernel
	   thread structure will be valid. */
	{ uint32 jif2 = jiffies+2;
		while (jiffies < jif2)
			;
	}

	thd_enabled = 1;
	
	/* Register a service */
	thd_svc_init();
	
	return 0;
}

/* Shutdown */
int thd_shutdown() {
	kthread_t *cur, *first;

	/* Remove the service */
	svcmpx_remove_handler("thread");

	/* Disable threading */
	irq_set_handler(EXC_TMU0_TUNI0, NULL);

	/* Kill remaining live threads */
	cur = first = thd_current;
	do {
		void *tmp = cur;
		cur = cur->next;
		free(tmp);
	} while (cur != first);

	thd_enabled = 0;
	
	return 0;
}



