/* KallistiOS 0.6

   jam.h
   (c)2000 Dan Potter

   $Id: jam.h,v 1.2 2000/11/09 05:20:03 bard Exp $
*/

#ifndef __JAM_H
#define __JAM_H

/* Interfaces imported from KallistiOS */
#include <kallisti/abi/thread.h>
#include <kallisti/abi/maple.h>
#include <kallisti/abi/ta.h>
#include <kallisti/abi/video.h>

extern abi_thread_t *thd;
extern abi_maple_t *maple;
extern abi_ta_t *ta;
extern abi_video_t *video;



/* JAM event types */
#define JAM_EVT_NONE		0		/* No event */

#define JAM_EVT_CURSORMOVE	0x10		/* Cursor moved */

#define JAM_EVT_BTNPRESS	0x20		/* Button press */
#define JAM_EVT_BTNRELEASE	0x21		/* Button release */
#define JAM_EVT_BTNANALOG	0x22		/* Button analog change */

#define JAM_EVT_KEYPRESS	0x30		/* Keyboard key pressed */

#define JAM_EVT_WNDCREATE	0x100		/* Window created */
#define JAM_EVT_WNDDESTROY	0x101		/* Window destroyed */

#define JAM_EVT_FOCUS		0x110		/* Window gained focus */
#define JAM_EVT_UNFOCUS		0x111		/* Window lost focus */

/* Button definitions */
#define JAM_BTN_FIRST		0x101
#define JAM_BTN_1		0x101		/* Left Mouse / A */
#define JAM_BTN_2		0x102		/* Side Mouse / X */
#define JAM_BTN_3		0x103		/* Right Mouse / B */
#define JAM_BTN_4		0x104		/* Y */
#define JAM_BTN_LAST		0x104
#define JAM_BTN_CNT		(JAM_BTN_LAST - JAM_BTN_FIRST + 1)

/* Redraw phases */
#define JAM_PHASE_NONE		0
#define JAM_PHASE_OPAQUE_POLY	1
#define	JAM_PHASE_OPAQUE_MOD	2
#define JAM_PHASE_TRANS_POLY	3
#define JAM_PHASE_TRANS_MOD	4
#define JAM_PHASE_PUNCHTHRU	5


/* Event structure: This is a generic event structure that tries to
   encompass just about everything. We're not expecting a massively
   complex window system so this ought to cover most of it. Unused members
   ought to be set to zero. */
typedef struct {
	uint32		type;		/* Event type */
	uint32		x, y;		/* Window-relative coordinate where it happened */
	uint32		button;		/* Button ID */
	uint32		analog;		/* Associated analog value */
	uint32		key,mod;	/* Related keypress and modifier */
} jam_event_t;


/* Window structure */
typedef struct jam_window_str {
	int	x1, y1;		/* Window location and size (user pane) */
	int	w, h;
	
	int	rx1, ry1;	/* Real window location and size (including decorations) */
	int	rw, rh;
	
	float	z1;		/* Z order (larger == higher) */
	uint32	flags;		/* Flag bits */

	char	title[64];	/* Window title */
	
	/* Redraw callback */	
	int	(*redraw)(struct jam_window_str *wnd, int phase);
	
	/* Event callback */
	int	(*event)(struct jam_window_str *wnd, jam_event_t *evt);
	
	struct jam_window_str *next;		/* Next window */
} jam_window_t;


/* Window Manager interface */
typedef struct {
	/* Set the "real" size for a given window */
	int	(*decor_set_sizes)(jam_window_t *wnd);

	/* Decorate a window (this uses the same phase codes as redraw() above) */
	int	(*decorate)(jam_window_t *wnd, int phase);

	/* Process a window event (inside decoration space) */
	int	(*decor_event)(jam_window_t *wnd, jam_event_t *evt);	
} jam_window_manager_t;


/****************************************************** Functions/Vars in jam.c ******/
extern volatile int jam_exit_flag;	/* Set to 1 to make JAM quit */

extern jam_window_manager_t *jam_mgr;	/* Window Manager (if any) */

extern uint32 util_texture;		/* Util texture pointer */

/* Set a window manager */
int jam_set_window_manager(jam_window_manager_t *mgr);

/* Allocate texture ram */
uint32 jam_texture_alloc(uint32 size);

/****************************************************** Functions/Vars in event.c ****/

/* Translate an event to the given window */
int jam_event_translate(jam_window_t *wnd, jam_event_t *evt);

/* Synthesize an event on the given window */
int jam_event_synth(jam_window_t *wnd, jam_event_t *evt);


/****************************************************** Functions/Vars in input.c ****/

extern int jam_ptr_x, jam_ptr_y;		/* Pointer location */

/* Lock the input to the given window */
int jam_input_lock(jam_window_t *owner);

/* Free the input */
int jam_input_unlock();

/* Check for new input on the Maple bus */
int jam_input_check();

/* The main maple loop (threadable) */
void jam_input_loop();


/****************************************************** Functions/Vars in repaint.c ***/
/* Draw one font character (12x24); doesn't do the poly header */
void jam_draw_char(float x1, float y1, float z1, float a, float r,
	float g, float b, int c);

/* Draw a string (12x24 font); does the polygon header */
void jam_draw_string(float x, float y, float z, float a, float r,
	float g, float b, char *str);

/* Draw a single colored box */
void jam_draw_box(float x1, float y1, float x2, float y2, float z,
		float a, float r, float g, float b);

/* Do a complete video frame. Note: This will cause a pause for
   vertical blank. */
void jam_repaint();


/****************************************************** Functions/Vars in window.c ***/

/* Window flags for jam_win_create */
#define JAM_WND_NONE		0x00000000	/* No flags */
#define JAM_WND_SHAPED		0x00000001	/* Shaped window: no border */
#define JAM_WND_NODECOR		0x00000002	/* No decorations */
#define JAM_WND_ONTOP		0x00000004	/* Always on top */
#define JAM_WND_ONBOTTOM	0x00000008	/* Always on bottom */

/* Only for internal JAM usage */
extern jam_window_t *jam_windows;		/* Linked list of active windows */
extern thd_mutex_t jam_windows_mutex;		/* Window list access mutex */

/* Finds the highest Z of any active window (for stacking) */
float jam_win_highest_z();

/* Restacks the given window to the highest Z order */
int jam_win_restack_top(jam_window_t *wnd);

/* Finds the active window which has the highest Z and encompasses
   the given x,y coordinates */
jam_window_t *jam_win_find_event(int x, int y);

/* Adds an active window */
jam_window_t *jam_win_create(int x1, int y1, int w, int h,
		int (*redraw_callback)(jam_window_t *wnd, int phase),
		int (*event_callback)(jam_window_t *wnd, jam_event_t *),
		char *title, uint32 flags);

/* Destroy an active window */
int jam_win_destroy(jam_window_t *wnd);

/* Move an active window */
int jam_win_move(jam_window_t *wnd, int x1, int y1);

#endif	/* __JAM_H */





