/* KallistiOS 0.6

   window.c
   (c)2000 Dan Potter

*/

static char id[] = "KOS $Id: window.c,v 1.1 2000/11/09 05:18:38 bard Exp $";


#include <malloc.h>
#include <string.h>
#include "jam.h"



/* List of active windows and a thread mutex for accessing it */
jam_window_t *jam_windows;
thd_mutex_t jam_windows_mutex;


/* Finds the highest Z of any active window (for stacking) */
float jam_win_highest_z() {
	jam_window_t *cur;
	float highz = 2.0f;
	
	thd->mutex_lock(&jam_windows_mutex);
	cur = jam_windows;
	while (cur != NULL) {
		if (cur->z1 > highz)
			highz = cur->z1;
		cur = cur->next;
	}
	thd->mutex_unlock(&jam_windows_mutex);
	
	return highz;
}

/* Restacks the given window to the highest Z order */
int jam_win_restack_top(jam_window_t *wnd) {
	wnd->z1 = jam_win_highest_z() + 1.0f;
	return 0;
}

/* Finds the active window which has the highest Z and encompasses
   the given x,y coordinates */
jam_window_t *jam_win_find_event(int x, int y) {
	jam_window_t *cur, *pick = NULL;
	float highz = 0.0f;
	
	thd->mutex_lock(&jam_windows_mutex);
	cur = jam_windows;
	while (cur != NULL) {
		if (x < cur->rx1 || y < cur->ry1
			|| x >= (cur->rx1+cur->rw)
			|| y >= (cur->ry1+cur->rh)) {
		} else {
			if (cur->z1 > highz) { highz = cur->z1; pick = cur; };
		}
		cur = cur->next;
	}
	thd->mutex_unlock(&jam_windows_mutex);
	
	return pick;
}

/* Adds an active window */
jam_window_t *jam_win_create(int x1, int y1, int w, int h,
		int (*redraw_callback)(jam_window_t *wnd, int phase),
		int (*event_callback)(jam_window_t *wnd, jam_event_t *),
		char *title, uint32 flags) {
	jam_window_t *n;

	/* Create a new window struct */	
	n = malloc(sizeof(jam_window_t));
	n->x1 = x1;
	n->y1 = y1;
	n->w = w;
	n->h = h;
	n->z1 = jam_win_highest_z() + 1.0f;
	strcpy(n->title, title);
	n->flags = flags;
	n->redraw = redraw_callback;
	n->event = event_callback;
		
	/* Add it to the list */
	thd->mutex_lock(&jam_windows_mutex);
	n->next = jam_windows;
	jam_windows = n;
	thd->mutex_unlock(&jam_windows_mutex);

	/* Readjust its size */
	if (jam_mgr) {
		jam_mgr->decor_set_sizes(n);
	}

	/* Send a create event */
	{
		jam_event_t event = { JAM_EVT_WNDCREATE };
		jam_event_synth(n, &event);
	}
	
	return 0;
}


/* Destroy an active window */
int jam_win_destroy(jam_window_t *wnd) {
	jam_window_t *cur, *last = NULL;
	
	thd->mutex_lock(&jam_windows_mutex);
	cur = jam_windows;
	while (cur != NULL) {
		if (cur == wnd) {
			if (last == NULL)
				jam_windows = cur->next;
			else
				last->next = cur->next;
			cur->next = NULL;
			break;
		}
		last = cur; cur = cur->next;
	}
	thd->mutex_unlock(&jam_windows_mutex);

	/* Send a destroy event and free the mem */
	if (cur != NULL) {
		{
			jam_event_t event = { JAM_EVT_WNDDESTROY };
			jam_event_synth(cur, &event);
		}
		free(cur);
	}
	
	return cur != NULL;
}

/* Move an active window */
int jam_win_move(jam_window_t *wnd, int x1, int y1) {
	wnd->x1 = x1;
	wnd->y1 = y1;

	/* Readjust its position */
	if (jam_mgr)
		jam_mgr->decor_set_sizes(wnd);
	return 0;
}




