/* KallistiOS 0.6

   Host-side serial console handler
   (c)2000 Dan Potter

   $Id: console.c,v 1.3 2000/11/12 01:09:00 bard Exp $
*/

static char id[] = "KOS $Id: console.c,v 1.3 2000/11/12 01:09:00 bard Exp $";

#include <fcntl.h>
#include <dirent.h>
#include <sys/stat.h>

#include <malloc.h>
#include <stdio.h>
#include "serial.h"

typedef unsigned long uint32;

/******************************************** Some STDIO functions */
int read_char() {
	char buf[1];
	while (read(0, buf, 1) < 1)
		;
	return buf[0];
}

void write_char(int c) {
	char buf[1] = {c};
	while (write(1, buf, 1) < 1)
		;
}

int read_long() {
	char buf[9] = {0};
	int i;
	for (i=0; i<8; i++)
		buf[i] = read_char();
	return strtol(buf, NULL, 16);
}


/******************************************** Serial helper functions */
/* Available console commands */
#define CMD_HELLO	0x48454c4f	/* HELO */
#define CMD_FIRST	1
#define CMD_PRINTK	1
#define CMD_DONE	2
#define CMD_OPENDIR	3
#define CMD_READDIR	4
#define CMD_CLOSEDIR	5
#define CMD_OPENFILE	6
#define CMD_CLOSEFILE	7
#define CMD_READFILE	8
#define CMD_TELLFILE	9
#define CMD_TOTALFILE	10
#define CMD_WRITEFILE	11
#define CMD_LAST	11

/* Available console responses */
#define RESP_OK		0
#define RESP_ERROR	0xffffffff

/* Represents one command packet */
typedef struct {
	uint32	command;
	uint32	datasize;
	uint32	xorsum;
	void	*data;
} packet_t;

/* Block until a char is available */
static int read_block() {
	return ser_getch();
}

/* Write a uint32 */
static void write_uint32(uint32 v) {
	ser_putch((v >> 24) & 0xff);
	ser_putch((v >> 16) & 0xff);
	ser_putch((v >> 8) & 0xff);
	ser_putch((v >> 0) & 0xff);
}

/* Read a uint32 (block until complete) */
static uint32 read_uint32() {
	uint32 v;
	
	v  = read_block() << 24;
	v |= read_block() << 16;
	v |= read_block() << 8;
	v |= read_block() << 0;
	
	return v;
}

/* Calculates the xorsum of a packet */
static uint32 xorsum_data(void *data, int size) {
	uint32 *d = (uint32*)data;
	uint32 sum = 0;
	
	if (size % 4) size += 4 - (size % 4);
	
	while (size-- > 0) {
		sum ^= *d++;
	}
	
	return sum;
}

/* Writes a complete command packet to the host software */
extern int serfd;
static int write_packet(packet_t *p) {
	uint32 ack, i;
	
	write_uint32(p->command);
	ack = read_uint32();
	if (ack != RESP_OK) return -1;

//	p->xorsum = xorsum_data(p->data, p->datasize);

	write_uint32(p->datasize);
	write_uint32(p->xorsum);
	
	for (i=0; i<p->datasize; i++) {
		ser_putch(((char*)p->data)[i]);
	}
	
	tcdrain(serfd);
	ack = read_uint32();
	if (ack != RESP_OK) return -1;
	
	return 0;
}

/******************************************** File commands */

/* Open current directory */
void file_opendir(packet_t *pkt) {
	char fn[256] = ".";
	DIR *dir;
	
	strncat(fn, (char*)pkt->data, 254);
	fn[255] = 0;
	
	dir = opendir(fn);
	if (dir == NULL) {
		write_uint32(RESP_ERROR);
	} else {
		packet_t pkt = {RESP_OK, 4, 0, &dir};
		write_uint32(RESP_OK);
		write_packet(&pkt);
	}
}

/* Read directory */
typedef struct {
	int 	size;
	char	name[256];
	uint32	time;
	uint32	attr;
} fs_dirent_t;
void file_readdir(packet_t *pkt) {
	DIR *dir = *((DIR**)pkt->data);
	if (dir == NULL) {
		write_uint32(RESP_ERROR);
	} else {
		fs_dirent_t fde;
		struct dirent *de;
		struct stat buf;
		packet_t pkt = { RESP_OK, sizeof(fs_dirent_t), 0, &fde };
		
		de = readdir(dir);
		if (de == NULL) {
			write_uint32(RESP_ERROR);
			return;
		}

		/* readdir only gave us the filename, now we get the
		   reset with stat() */
		if (stat(de->d_name, &buf) < 0) {
			write_uint32(RESP_ERROR);
			return;
		}
		write_uint32(RESP_OK);
		
		if (S_ISDIR(buf.st_mode))
			fde.size = -1;
		else
			fde.size = buf.st_size;
		strncpy(fde.name, de->d_name, 255); fde.name[255] = 0;
		fde.time = buf.st_mtime;
		fde.attr = 0;
		
		write_packet(&pkt);
	}
}

/* Close directory */
void file_closedir(packet_t *pkt) {
	DIR *dir = *((DIR**)pkt->data);
	if (dir == NULL) {
		write_uint32(RESP_ERROR);
	} else {
		closedir(dir);
		write_uint32(RESP_OK);
	}
}

/* Open file */
void file_open(packet_t *pkt) {
	int mode = *((uint32*)pkt->data);
	char fn[256] = ".";
	int fd;
	
	strncat(fn, ((char*)pkt->data)+4, 254);
	fn[255] = 0;

	if ((mode & 7) == 4) {
		printf("KONSOLE: opening for write\n");
		fd = open(fn, O_WRONLY | O_CREAT);
	}
	else if ((mode & 7) == 1) {
		printf("KONSOLE: opening for read\n");
		fd = open(fn, O_RDONLY);
	}
	else {
		printf("KONSOLE: Unsupported open mode %04x\n", mode);
		fd = -1;
	}
	if (fd < 0) {
		perror("KONSOLE: can't open file:");
		write_uint32(RESP_ERROR);
	}
	else {
		packet_t pkt = {RESP_OK, 4, 0, &fd};
		write_uint32(RESP_OK);
		write_packet(&pkt);
	}
}

/* Close file */
void file_close(packet_t *pkt) {
	int fd = *((int*)pkt->data);
	
	write_uint32(RESP_OK);
	close(fd);
}

/* Read file */
void file_read(packet_t *pkt) {
	int fd = *((int *)pkt->data);
	int size = *((int *)(pkt->data+4));
	void *buffer;
	int r;

	printf("KONSOLE: Sending %d bytes\n", size);
	buffer = malloc(size);
	if (buffer == NULL) {
		write_uint32(RESP_ERROR);
		return;
	}
	r = read(fd, buffer, size);
	if (r < 0) {
		write_uint32(RESP_ERROR);
		return;
	}
	write_uint32(RESP_OK);
	
	pkt->command = RESP_OK;
	pkt->datasize = r;
	pkt->xorsum = 0;
	pkt->data = buffer;
	write_packet(pkt);
	
	free(buffer);
}

/* Write file */
void file_write(packet_t *pkt) {
	int fd = *((int *)pkt->data);
	int size = *((int *)(pkt->data+4));
	void *buffer = pkt->data+8;
	int r;
	
	printf("KONSOLE: Receiving %d bytes\n", size);
	r = write(fd, buffer, size);
	if (r < 0) {
		write_uint32(RESP_ERROR);
		return;
	}
	write_uint32(RESP_OK);
	
	pkt->command = RESP_OK;
	pkt->datasize = 4;
	pkt->xorsum = 0;
	pkt->data = &r;
	write_packet(pkt);
}

/******************************************** Main stuff */

/* Synchronize the serial stream with the target */
void sync_to_target() {
	int c;
	
	while(1) {
		/* Look for the first char (H)*/
		while (ser_getch() != 'H')
			;
	
		/* Next is the E (heh)*/
		c = ser_getch();
		if (c != 'E') continue;
		
		/* Next (L) */
		c = ser_getch();
		if (c != 'L') continue;
		
		/* Next (O) */
		c = ser_getch();
		if (c != 'O') continue;
		
		ser_putch('O');
		break;
	}
	
	fprintf(stderr, "Connected with target\n");
}

/* Process an individual packet */
void process_pkt(packet_t *pkt) {
	switch(pkt->command) {
		case CMD_PRINTK:	/* printk from kernel */
			fprintf(stdout, "%s", pkt->data);
			fflush(stdout);
			write_uint32(RESP_OK);
			break;
		case CMD_DONE:		/* Should never get this one here */
			break;
		case CMD_OPENDIR:	/* Open directory for list*/
			printf("KONSOLE: Received OPENDIR\n");
			file_opendir(pkt);
			break;
		case CMD_READDIR:	/* Read directory */
			printf("KONSOLE: Received READDIR\n");
			file_readdir(pkt);
			break;
		case CMD_CLOSEDIR:	/* Close directory */
			printf("KONSOLE: Received CLOSEDIR\n");
			file_closedir(pkt);
			break;
		case CMD_OPENFILE:	/* Open file */
			printf("KONSOLE: Received OPENFILE\n");
			file_open(pkt);
			break;
		case CMD_CLOSEFILE:	/* Close file */
			printf("KONSOLE: Received CLOSEFILE\n");
			file_close(pkt);
			break;
		case CMD_READFILE:	/* Read file */
			printf("KONSOLE: Received READFILE\n");
			file_read(pkt);
			break;
		case CMD_TELLFILE:	/* Tell file */
			printf("KONSOLE: Received TELLFILE\n");
			break;
		case CMD_TOTALFILE:	/* Total file */
			printf("KONSOLE: Received TOTALFILE\n");
			break;
		case CMD_WRITEFILE:	/* Write file */
			printf("KONSOLE: Received WRITEFILE\n");
			file_write(pkt);
			break;
	}
}

/* Main loop: process all incoming commands */
void main_loop() {
	packet_t pkt;
	int i, cnt = 0;
	void *buffer;
	
	while(cnt < 10) {
		/* Read a command word */
		pkt.command = read_uint32();
		//printf("Received command %08lx\n", pkt.command);
		if (pkt.command < CMD_FIRST || pkt.command > CMD_LAST) {
			printf("KONSOLE: Unrecognized command %08lx\n", pkt.command);
			write_uint32(0xffffffff);
			cnt++;
			continue;
		}
		write_uint32(RESP_OK);
		cnt = 0;
		
		/* Read its data size and xorsum */
		//printf("Reading datasize\n");
		pkt.datasize = read_uint32();
		//printf("ds=%d, reading xorsum\n", pkt.datasize);
		pkt.xorsum = read_uint32();

		//printf("Reading %d bytes, xorsum %08lx\n", pkt.datasize, pkt.xorsum);
		
		/* Now read the data */
		pkt.data = buffer = malloc(pkt.datasize);
		for (i=0; i<pkt.datasize; i++) {
			((char*)pkt.data)[i] = ser_getch();
		}
				
		/* Process the packet */
		if (pkt.command == CMD_DONE) return;
		process_pkt(&pkt);
		free(buffer);
	}
	
	printf("KONSOLE: Too many unknown commands, quitting\n");
}

void console() {
	/* Open the serial port */
	ser_init("/dev/ttyS1", 57600);
	
	/* Sync with the target */
	sync_to_target();
	
	/* Enter the main console loop */
	main_loop();
	
	/* Reset dc-load */
	ser_putch('x');
	ser_putch('x');
	ser_putch('x');
	ser_putch('x');
	
	/* Empty buffer */
	while(1)
		ser_getch();
}




